import os, json
from flask import Blueprint, request, jsonify
from openai import OpenAI
from dotenv import load_dotenv

load_dotenv()
ai = OpenAI(api_key=os.getenv("OPENAI_API_KEY"))
CLIENT_TOKEN = os.getenv("TV_TOKEN", "CHANGE_ME")

tradingview_bp = Blueprint("tradingview", __name__)

SYSTEM_PROMPT = """أنت محلل تداول... (نفس البرومبت اللي وريتك إياه قبل)"""

def build_user_prompt(payload: dict) -> str:
    return (
        f"الرمز: {payload.get('ticker')}\n"
        f"الإطار: {payload.get('interval')}\n"
        f"OHLC: o={payload.get('o')} h={payload.get('h')} l={payload.get('l')} c={payload.get('c')}\n"
        f"EMA20/50/200: {payload.get('ema20')}, {payload.get('ema50')}, {payload.get('ema200')}\n"
        f"RSI: {payload.get('rsi')}\n"
        f"MACD/signal/hist: {payload.get('macd')}/{payload.get('macdSignal')}/{payload.get('macdHist')}\n"
        f"أحداث: {payload.get('events')}\n"
    )

@tradingview_bp.route("/webhook", methods=["POST"])
def tradingview_webhook():
    token = request.args.get("token")
    if token != CLIENT_TOKEN:
        return jsonify({"error": "unauthorized"}), 401

    try:
        payload = request.get_json(force=True, silent=True)
        if payload is None:
            payload = json.loads(request.data.decode("utf-8"))
    except Exception:
        return jsonify({"error": "invalid json"}), 400

    try:
        user_prompt = build_user_prompt(payload)
        completion = ai.chat.completions.create(
            model="gpt-4o-mini",
            response_format={"type": "json_object"},
            messages=[
                {"role": "system", "content": SYSTEM_PROMPT},
                {"role": "user", "content": user_prompt}
            ],
            temperature=0.2
        )
        result = json.loads(completion.choices[0].message.content)
    except Exception as e:
        result = {
            "direction": "sideways",
            "confidence": 50,
            "riskLevel": "متوسط",
            "timeFrame": payload.get("interval"),
            "entryPrice": payload.get("c"),
            "supportLevels": [payload.get("ll")],
            "resistanceLevels": [payload.get("hh")],
            "reasons": [f"Fallback: {str(e)}"]
        }

    return jsonify(result), 200
