from flask import Blueprint, request, jsonify
from datetime import datetime, timedelta
import random
from src.models.opportunity import db, Opportunity

opportunities_bp = Blueprint('opportunities', __name__)

@opportunities_bp.route('/live', methods=['GET'])
def get_live_opportunities():
    """Get current live opportunities"""
    try:
        # Get live opportunities from database
        opportunities = Opportunity.query.filter_by(is_live=True).filter(
            Opportunity.entry_time > datetime.utcnow()
        ).order_by(Opportunity.created_at.desc()).limit(3).all()
        
        # If no opportunities in database, generate mock data
        if not opportunities:
            opportunities = generate_mock_opportunities()
        
        return jsonify({
            'opportunities': [opp.to_dict() for opp in opportunities],
            'last_update': datetime.utcnow().isoformat(),
            'next_update': (datetime.utcnow() + timedelta(seconds=10)).isoformat()
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب الفرص'}), 500

@opportunities_bp.route('/history', methods=['GET'])
def get_opportunities_history():
    """Get historical opportunities with pagination"""
    try:
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 10, type=int)
        
        opportunities = Opportunity.query.filter_by(is_live=False).order_by(
            Opportunity.created_at.desc()
        ).paginate(
            page=page, 
            per_page=per_page, 
            error_out=False
        )
        
        return jsonify({
            'opportunities': [opp.to_dict() for opp in opportunities.items],
            'total': opportunities.total,
            'pages': opportunities.pages,
            'current_page': page,
            'per_page': per_page
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب السجل'}), 500

def generate_mock_opportunities():
    """Generate mock opportunities for demonstration"""
    pairs = ['EUR/USD', 'GBP/USD', 'USD/JPY', 'AUD/USD', 'USD/CAD', 'NZD/USD']
    directions = ['up', 'down']
    risk_levels = ['منخفض', 'متوسط', 'عالي']
    
    reasons = [
        'كسر مستوى المقاومة مع حجم تداول عالي',
        'ارتداد من مستوى الدعم القوي',
        'تشكيل نموذج فني صاعد على الشارت',
        'اختراق خط الاتجاه مع تأكيد المؤشرات',
        'ارتداد من مقاومة قوية مع تشكيل نموذج هابط',
        'كسر مستوى دعم مهم مع زيادة في الحجم'
    ]
    
    mock_opportunities = []
    
    for i in range(3):
        entry_time = datetime.utcnow() + timedelta(minutes=random.randint(1, 10))
        
        opportunity = Opportunity(
            pair_name=random.choice(pairs),
            direction=random.choice(directions),
            time_frame='5 دقائق',
            marginal=round(random.uniform(1.0, 3.0), 1),
            reason=random.choice(reasons),
            confidence=random.randint(70, 95),
            risk_level=random.choice(risk_levels),
            entry_time=entry_time,
            is_live=True
        )
        
        mock_opportunities.append(opportunity)
    
    return mock_opportunities

@opportunities_bp.route('/create', methods=['POST'])
def create_opportunity():
    """Create a new opportunity (Admin only)"""
    try:
        data = request.get_json()
        
        # Validate required fields
        required_fields = ['pair_name', 'direction', 'time_frame', 'marginal', 'reason', 'confidence', 'risk_level', 'entry_time']
        for field in required_fields:
            if field not in data:
                return jsonify({'message': f'الحقل {field} مطلوب'}), 400
        
        # Parse entry time
        entry_time = datetime.fromisoformat(data['entry_time'].replace('Z', '+00:00'))
        
        opportunity = Opportunity(
            pair_name=data['pair_name'],
            direction=data['direction'],
            time_frame=data['time_frame'],
            marginal=float(data['marginal']),
            reason=data['reason'],
            confidence=int(data['confidence']),
            risk_level=data['risk_level'],
            entry_time=entry_time,
            is_live=data.get('is_live', True),
            created_by=data.get('created_by')
        )
        
        db.session.add(opportunity)
        db.session.commit()
        
        return jsonify({
            'message': 'تم إنشاء الفرصة بنجاح',
            'opportunity': opportunity.to_dict()
        }), 201
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في إنشاء الفرصة'}), 500

@opportunities_bp.route('/<opportunity_id>', methods=['PUT'])
def update_opportunity(opportunity_id):
    """Update an opportunity (Admin only)"""
    try:
        opportunity = Opportunity.query.get(opportunity_id)
        if not opportunity:
            return jsonify({'message': 'الفرصة غير موجودة'}), 404
        
        data = request.get_json()
        
        # Update fields
        for field in ['pair_name', 'direction', 'time_frame', 'marginal', 'reason', 'confidence', 'risk_level', 'is_live']:
            if field in data:
                if field == 'entry_time':
                    setattr(opportunity, field, datetime.fromisoformat(data[field].replace('Z', '+00:00')))
                else:
                    setattr(opportunity, field, data[field])
        
        db.session.commit()
        
        return jsonify({
            'message': 'تم تحديث الفرصة بنجاح',
            'opportunity': opportunity.to_dict()
        }), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في تحديث الفرصة'}), 500

@opportunities_bp.route('/<opportunity_id>', methods=['DELETE'])
def delete_opportunity(opportunity_id):
    """Delete an opportunity (Admin only)"""
    try:
        opportunity = Opportunity.query.get(opportunity_id)
        if not opportunity:
            return jsonify({'message': 'الفرصة غير موجودة'}), 404
        
        db.session.delete(opportunity)
        db.session.commit()
        
        return jsonify({'message': 'تم حذف الفرصة بنجاح'}), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في حذف الفرصة'}), 500

