from flask import Blueprint, request, jsonify
import random
from datetime import datetime

dashboard_bp = Blueprint('dashboard', __name__)

@dashboard_bp.route('/live-data', methods=['GET'])
def get_live_data():
    """Get live market data for dashboard"""
    try:
        # Generate mock live data (in real implementation, this would connect to market data feeds)
        pairs_data = generate_mock_market_data()
        
        return jsonify({
            'pairs': pairs_data,
            'market_status': get_market_status(),
            'last_update': datetime.utcnow().isoformat(),
            'next_update': 10  # seconds
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب البيانات المباشرة'}), 500

def generate_mock_market_data():
    """Generate mock market data"""
    pairs = [
        {'pair': 'EUR/USD', 'base_price': 1.0856},
        {'pair': 'GBP/USD', 'base_price': 1.2734},
        {'pair': 'USD/JPY', 'base_price': 149.82},
        {'pair': 'AUD/USD', 'base_price': 0.6523},
        {'pair': 'USD/CAD', 'base_price': 1.3678},
        {'pair': 'NZD/USD', 'base_price': 0.5987}
    ]
    
    volumes = ['0.8M', '1.2M', '1.8M', '2.3M', '3.1M', '1.9M']
    
    pairs_data = []
    for i, pair_info in enumerate(pairs):
        # Generate realistic price movement
        change = random.uniform(-0.02, 0.02)
        current_price = pair_info['base_price'] + change
        change_percent = (change / pair_info['base_price']) * 100
        
        trend = 'up' if change > 0 else 'down'
        strength = random.randint(55, 95)
        
        pairs_data.append({
            'pair': pair_info['pair'],
            'price': round(current_price, 4 if 'JPY' not in pair_info['pair'] else 2),
            'change': round(change, 4),
            'change_percent': round(change_percent, 2),
            'trend': trend,
            'strength': strength,
            'volume': volumes[i]
        })
    
    return pairs_data

def get_market_status():
    """Get current market status"""
    # Simulate market conditions
    statuses = ['نشط', 'هادئ', 'متقلب']
    volatility_levels = ['منخفض', 'متوسط', 'عالي']
    opportunity_levels = ['ضعيف', 'جيد', 'ممتاز']
    
    return {
        'status': random.choice(statuses),
        'volatility': random.choice(volatility_levels),
        'opportunities': random.choice(opportunity_levels),
        'active_pairs': random.randint(15, 25),
        'trending_up': random.randint(8, 15),
        'trending_down': random.randint(5, 12)
    }

@dashboard_bp.route('/statistics', methods=['GET'])
def get_statistics():
    """Get trading statistics"""
    try:
        # In real implementation, these would be calculated from actual data
        stats = {
            'total_opportunities': random.randint(150, 300),
            'successful_trades': random.randint(120, 250),
            'win_rate': round(random.uniform(75, 95), 1),
            'average_profit': round(random.uniform(15, 35), 1),
            'total_profit': random.randint(500, 1500),
            'active_users': random.randint(25, 45),
            'daily_volume': f"{random.randint(50, 150)}M",
            'market_coverage': random.randint(85, 98)
        }
        
        return jsonify(stats), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب الإحصائيات'}), 500

@dashboard_bp.route('/alerts', methods=['GET'])
def get_alerts():
    """Get system alerts and notifications"""
    try:
        alerts = [
            {
                'id': 1,
                'type': 'info',
                'title': 'تحديث النظام',
                'message': 'تم تحديث خوارزميات التحليل لتحسين دقة الإشارات',
                'timestamp': datetime.utcnow().isoformat(),
                'read': False
            },
            {
                'id': 2,
                'type': 'warning',
                'title': 'تحذير السوق',
                'message': 'توقع تقلبات عالية بسبب الأخبار الاقتصادية المهمة',
                'timestamp': datetime.utcnow().isoformat(),
                'read': False
            }
        ]
        
        return jsonify({'alerts': alerts}), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب التنبيهات'}), 500

