from flask import Blueprint, request, jsonify
import os
import uuid
import json
import random
from datetime import datetime
from werkzeug.utils import secure_filename
from src.models.chart_analysis import db, ChartAnalysis

chart_analysis_bp = Blueprint('chart_analysis', __name__)

UPLOAD_FOLDER = 'uploads/charts'
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg', 'gif'}

def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

def ensure_upload_folder():
    """Ensure upload folder exists"""
    upload_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), UPLOAD_FOLDER)
    os.makedirs(upload_path, exist_ok=True)
    return upload_path

@chart_analysis_bp.route('/upload', methods=['POST'])
def upload_chart():
    """Upload and analyze chart image"""
    try:
        # Check if file is present
        if 'chart' not in request.files:
            return jsonify({'message': 'لم يتم رفع أي ملف'}), 400
        
        file = request.files['chart']
        if file.filename == '':
            return jsonify({'message': 'لم يتم اختيار ملف'}), 400
        
        if not allowed_file(file.filename):
            return jsonify({'message': 'نوع الملف غير مدعوم. يرجى رفع صورة PNG, JPG, أو JPEG'}), 400
        
        # Save file
        upload_path = ensure_upload_folder()
        filename = secure_filename(f"{uuid.uuid4()}_{file.filename}")
        file_path = os.path.join(upload_path, filename)
        file.save(file_path)
        
        # Simulate chart analysis (in real implementation, this would use AI/ML)
        analysis_result = simulate_chart_analysis()
        
        # Save analysis to database
        chart_analysis = ChartAnalysis(
            user_id=request.form.get('user_id'),
            image_filename=filename,
            image_path=file_path,
            result_direction=analysis_result['direction'],
            confidence=analysis_result['confidence'],
            time_frame=analysis_result['time_frame'],
            marginal=analysis_result['marginal'],
            reasons=json.dumps(analysis_result['reasons']),
            risk_level=analysis_result['risk_level'],
            entry_price=analysis_result['entry_price'],
            target_price=analysis_result['target_price']
        )
        
        db.session.add(chart_analysis)
        db.session.commit()
        
        return jsonify({
            'message': 'تم تحليل الشارت بنجاح',
            'analysis': chart_analysis.to_dict()
        }), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في تحليل الشارت'}), 500

def simulate_chart_analysis():
    """Simulate AI chart analysis"""
    directions = ['up', 'down']
    risk_levels = ['منخفض', 'متوسط', 'عالي']
    
    reasons_up = [
        'تشكيل نموذج فني صاعد على الشارت',
        'كسر مستوى المقاومة مع حجم تداول عالي',
        'تأكيد من المؤشرات الفنية للاتجاه الصاعد',
        'ارتداد قوي من مستوى الدعم المهم'
    ]
    
    reasons_down = [
        'تشكيل نموذج فني هابط على الشارت',
        'ارتداد من مستوى المقاومة القوي',
        'كسر مستوى الدعم مع زيادة في الحجم',
        'تأكيد من المؤشرات الفنية للاتجاه الهابط'
    ]
    
    direction = random.choice(directions)
    confidence = random.randint(70, 95)
    
    base_price = round(random.uniform(1.0000, 1.5000), 4)
    
    if direction == 'up':
        entry_price = f"{base_price:.4f}"
        target_price = f"{base_price + random.uniform(0.0020, 0.0080):.4f}"
        reasons = random.sample(reasons_up, random.randint(2, 4))
    else:
        entry_price = f"{base_price:.4f}"
        target_price = f"{base_price - random.uniform(0.0020, 0.0080):.4f}"
        reasons = random.sample(reasons_down, random.randint(2, 4))
    
    return {
        'direction': direction,
        'confidence': confidence,
        'time_frame': '5 دقائق',
        'marginal': round(random.uniform(1.0, 3.0), 1),
        'reasons': reasons,
        'risk_level': random.choice(risk_levels),
        'entry_price': entry_price,
        'target_price': target_price
    }

@chart_analysis_bp.route('/history', methods=['GET'])
def get_analysis_history():
    """Get user's chart analysis history"""
    try:
        user_id = request.args.get('user_id')
        page = request.args.get('page', 1, type=int)
        per_page = request.args.get('per_page', 10, type=int)
        
        query = ChartAnalysis.query
        if user_id:
            query = query.filter_by(user_id=user_id)
        
        analyses = query.order_by(ChartAnalysis.created_at.desc()).paginate(
            page=page,
            per_page=per_page,
            error_out=False
        )
        
        return jsonify({
            'analyses': [analysis.to_dict() for analysis in analyses.items],
            'total': analyses.total,
            'pages': analyses.pages,
            'current_page': page,
            'per_page': per_page
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب سجل التحليلات'}), 500

@chart_analysis_bp.route('/<analysis_id>', methods=['GET'])
def get_analysis(analysis_id):
    """Get specific analysis details"""
    try:
        analysis = ChartAnalysis.query.get(analysis_id)
        if not analysis:
            return jsonify({'message': 'التحليل غير موجود'}), 404
        
        return jsonify({
            'analysis': analysis.to_dict()
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في جلب التحليل'}), 500

@chart_analysis_bp.route('/<analysis_id>', methods=['DELETE'])
def delete_analysis(analysis_id):
    """Delete analysis and associated image"""
    try:
        analysis = ChartAnalysis.query.get(analysis_id)
        if not analysis:
            return jsonify({'message': 'التحليل غير موجود'}), 404
        
        # Delete image file
        try:
            if os.path.exists(analysis.image_path):
                os.remove(analysis.image_path)
        except:
            pass  # Continue even if file deletion fails
        
        db.session.delete(analysis)
        db.session.commit()
        
        return jsonify({'message': 'تم حذف التحليل بنجاح'}), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في حذف التحليل'}), 500

