from flask import Blueprint, request, jsonify
from datetime import datetime
from src.models.user import db, User
from src.models.access_key import AccessKey

auth_bp = Blueprint('auth', __name__)

@auth_bp.route('/login', methods=['POST'])
def login():
    """Login with access key"""
    try:
        data = request.get_json()
        access_key = data.get('access_key', '').strip()
        
        if not access_key:
            return jsonify({'message': 'مفتاح الدخول مطلوب'}), 400
        
        # Find the access key
        key_obj = AccessKey.query.filter_by(key_value=access_key).first()
        
        if not key_obj:
            return jsonify({'message': 'مفتاح الدخول غير صحيح'}), 401
        
        # Check if key is valid
        if not key_obj.is_valid():
            return jsonify({'message': 'مفتاح الدخول منتهي الصلاحية أو غير نشط'}), 401
        
        # Update usage statistics
        key_obj.last_used = datetime.utcnow()
        key_obj.usage_count += 1
        
        # Create or update user if needed
        user = None
        if key_obj.user_id:
            user = User.query.get(key_obj.user_id)
        
        if not user:
            user = User()
            db.session.add(user)
            db.session.flush()  # Get the user ID
            key_obj.user_id = user.id
        
        user.last_login = datetime.utcnow()
        
        db.session.commit()
        
        return jsonify({
            'message': 'تم تسجيل الدخول بنجاح',
            'user': user.to_dict(),
            'access_key': {
                'expires_at': key_obj.expires_at.isoformat() if key_obj.expires_at else None,
                'usage_count': key_obj.usage_count
            }
        }), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({'message': 'حدث خطأ في الخادم'}), 500

@auth_bp.route('/verify', methods=['POST'])
def verify():
    """Verify access key validity"""
    try:
        data = request.get_json()
        access_key = data.get('access_key', '').strip()
        
        if not access_key:
            return jsonify({'message': 'مفتاح الدخول مطلوب'}), 400
        
        # Find the access key
        key_obj = AccessKey.query.filter_by(key_value=access_key).first()
        
        if not key_obj or not key_obj.is_valid():
            return jsonify({'message': 'مفتاح الدخول غير صالح'}), 401
        
        # Get user info
        user = None
        if key_obj.user_id:
            user = User.query.get(key_obj.user_id)
        
        return jsonify({
            'message': 'مفتاح الدخول صالح',
            'user': user.to_dict() if user else None,
            'expires_at': key_obj.expires_at.isoformat() if key_obj.expires_at else None
        }), 200
        
    except Exception as e:
        return jsonify({'message': 'حدث خطأ في الخادم'}), 500

@auth_bp.route('/logout', methods=['POST'])
def logout():
    """Logout (mainly for frontend state management)"""
    return jsonify({'message': 'تم تسجيل الخروج بنجاح'}), 200

